#ifndef MOTOR_H
#define MOTOR_H

typedef	enum{//MotorErrorState	
	Clear = 			0, 			//0000	0000
	OverVbus = 			1, 			//0000	0001
	UnderVbus = 		2, 			//0000	0010
	OverTemperature = 	4, 			//0000	0100
	FaultLock = 		8, 			//0000	1000
	AOCP = 				16, 		//0001	0000
	POCP = 				32			//0010	0000
}ErrorStatus;

enum MotorStatus{//MotorState
	M_OFF = 			0, 
	M_INIT = 			1, 
	M_TAILWIND = 		2, 
	M_IPD = 			3,
	M_Parking = 		4,
	M_START = 			5,
	M_RUN = 			6, 
	M_STOP = 			7, 
	M_ERROR = 			8,
	M_BMF_BREAK = 		9,
	M_ADC_INIT = 		10
};

#define Current_Control 0
#define Speed_Control 1
#define Power_Limit 2
#define Power_Control 3

//*******************************************************************************************************************************************
// <<< Use Configuration Wizard in Context Menu >>>
	
// <h> Set motor parameters
	// <o> Motor Pole <0-30:2>
		#define POLE 16
		#define POLE_PAIRS POLE>>1

	// <o> Motor SMO_G <0-32767>
	// <i> Formula : G = Ts / Ls/2
		#define G  2000//8000

	// <o> Motor SMO_F <0-32767>
	// <i> Formula : F = 1 - (G * Rs/2)
		#define F 31000//18000

	// <o> Motor SMO_Kslf <0-32767>
	// <i> Description : Low Pass Filter Gain : Kslf
		#define Kslf 6000

	// <o> Motor SMO_Z-Corr <0-32767>
		#define ZCorr 32767

	// <o> Motor SMO_BanGain (Sat) <0-32767>
		#define BanGain 32767

	// <o> Motor SMO_MaxSmcError (Limit) <0-32767>
		#define MaxSmcError 32767//32767
		
	// <o> Motor SMO_Gain (Q15(BanGain/MaxSmcError)) <0-32767>
		#define Kslide	32767//16383
// </h>

// <h> Set Rshunt and OPA Gain
	// <o> Rshunt(unit : 0.1mR)
		#define R_SHUNT (float)1000/10000 

	// <o.4..5> OPA GAIN <0=> 1 Gain <1=> 2.5 Gain <2=> 5 Gain <3=> 10 Gain
	// <i> Note : Need to be the same as ADCSTR
		#define OPA_GAIN_REGS 0x10
		#if (OPA_GAIN_REGS == 0x00)
			#define OPA_GAIN (float)10/10
		#endif
		#if (OPA_GAIN_REGS == 0x10)
			#define OPA_GAIN (float)25/10 
		#endif
		#if (OPA_GAIN_REGS == 0x20)
			#define OPA_GAIN (float)50/10 
		#endif
		#if (OPA_GAIN_REGS == 0x30)
			#define OPA_GAIN (float)100/10 
		#endif
		#define I_AMPLIFIER (uint16)((float)((((R_SHUNT) * (OPA_GAIN) * 1024))/5)*64)//655 
// </h>

// <h> Set the motor tuning process
	// <o> SOP_LEVEL <0=> LEVEL_0 <1=> LEVEL_1 <2=> LEVEL_2 
	// <i> LEVEL_0 : Current Opa correction (note :Fine tune Ia Ib Offset parameters)
	// <i> LEVEL_1 : OpenLoop (Note : Fine tune G,F,Kslide,Kslf parameters. After operation, you need to manually switch CloseLoopFlag)
	// <i> LEVEL_2 : CloseLoop(Note : Automatically switch CloseLoopFlag after operation)
		#define SOP_LEVEL 2
	// <e> DCR Start Strategy
		#define DCR_Strategy 0
		#if (DCR_Strategy == 1)
		// <o> Set DCR_V max values (unit:%) <0-100>
				#define	dDCR_V_max_Limit	15
			// <o> Set DCR_V min values (unit:%) <0-100>	
				#define	dDCR_V_min_Limit	5
			// <o> Set DCR time (unit:1ms) 
				#define	dDCR_time	1000*2
			// <o1> Set DCR Zone1 percentage of time (unit:%) 
				#define	dDCR_Zone1_percentage	(unsigned int)((float)24512/((dDCR_time)*(float)80/100))
			// <o1> Set DCR Zone2 percentage of time  (unit:%)
				#define	dDCR_Zone2_percentage	(unsigned int)((float)24512/((dDCR_time)*(float)20/100))
				
				#define	dDCR_V_max_values					(unsigned int)((float)(dDCR_V_max_Limit)*32767/100)	
				#define	dDCR_V_min_values					(unsigned int)((float)(dDCR_V_min_Limit)*32767/100)
		#endif
	// </e>	
	
	// <e> Square Wave Parking Start Strategy
		#define Square_Parking 0
		#if (Square_Parking == 1)
			// <o> Set Startup_duty <0-32767>
				#define Startup_duty 		256     
			// <o> Set Square Parking Time
			#define USER_START_DELAY 		500 
		#endif
	// </e>	
	
	#if (DCR_Strategy == 1) && (Square_Parking == 1)
		#error Wrong setting DCR_Strategy and Square_Parking !!!!
	#endif
			
	// <o> Set IQ parking duration(unit : ms)
	// <i> unit : mSec ex. 100 * 50us = 5ms
		#define I_ALIG_DURATION 1 // unit : mSec ex. 100 * 50us = 5ms
		
		#if ((SOP_LEVEL == 1) || (SOP_LEVEL == 2))
		// <o> Set SMO_PLL initial speed (unit : rpm)
			#define FREQ_FIRST_START_VALUE (short)((float) 1 * 32767 / (BASE_RPM)) // unit : Rpm
			//32767 / BASE_RPM = POLE_PAIRS
			
		// <o> Set SMO_PLL end speed (unit : rpm)
			#define FREQ_FINAL_START_VALUE (short)((float) 30 * 32767 / (BASE_RPM)) 
			//32767 / BASE_RPM = POLE_PAIRS
			
		// <o> Set PLL accumulation
			#define PLL_ACC 1
		// <o> Set SMO_RAMP acceleration slope (unit : ms) <0-10000> 
			#define SMO_RAMP 50
		#endif

	// <o> Set SMO_DELAY Delay time (unit : ms) <0-65535>
		#define SMO_DELAY_DURATION 5

// </h>

// <h> Set FOC LOOP Parameter
	// <h> IQ
		// <h> Set IQ Current parameter 
			// <o> Set IQ Initial current (unit : mA)
				#define IQ_INIT_START_VALUE (int16)((float) 1000/1000 * I_AMPLIFIER) // unit : A * Gain

			// <o> Set IQ Starting current (unit : mA)
				#define IQ_FIRST_START_VALUE (int16)((float) 1200/1000 * I_AMPLIFIER) // unit : A * Gain

			// <o> Set IQ End current (unit : mA)
				#define IQ_FINAL_START_VALUE (int16)((float) 1400/1000 * I_AMPLIFIER) // unit : A * Gain
				
				#define I_ALIG_STEP_VALUE ((float)((IQ_FIRST_START_VALUE) - (IQ_INIT_START_VALUE)) / (I_ALIG_DURATION))	
		// </h>
		
		// <h> Set IQ PI parameters
			// <o>  Kp parameters <0-32767>
				#define Iq_Kp 	8000
				
			// <o> Ki parameters <0-32767>
				#define Iq_Ki 	20
				
			// <o> Kt parameters <0-32767>
				#define Iq_Kt 	32767
				
			// <o> MaxLimit parameters <0-32767>
				#define Iq_MaxLimit 32767
				
			// <o> MinLimit parameters <0-32767>
				#define Iq_MinLimit -32767
		// </h>
	// </h>
	
	// <h> ID
		// <h> Set ID Current parameter
			// <o> Set ID Starting current (unit : mA)
				#define ID_FIRST_START_VALUE (int16)((float) -0/1000 * I_AMPLIFIER) // unit : A * Gain

			// <o> Set ID End current (unit : mA)
				#define ID_FINAL_START_VALUE (int16)((float) -0/1000 * I_AMPLIFIER) // unit : A * Gain
		// </h>
		
		// <h> Set ID PI parameters
			// <o>  Kp parameters <0-32767>
				#define Id_Kp 31000
				
			// <o> Ki parameters <0-32767>
				#define Id_Ki 20
				
			// <o> Kt parameters <0-32767>
				#define Id_Kt 32767
				
			// <o> MaxLimit parameters <0-32767>
				#define Id_MaxLimit 32767
				
			// <o> MinLimit parameters <0-32767>
				#define Id_MinLimit -32767
		// </h>
	// </h>
	
	// <h> SPEED
		// <h> Set SPEED PI parameters
			// <o> Start Kp parameters <0-32767>
				#define Spd_Kp_Start 	0
				
			// <o> Final Kp parameters <0-32767>
				#define Spd_Kp_Final 	25000
				
			// <o> Ki parameters <0-32767>
				#define Spd_Ki 			4
				
			// <o> Kt parameters <0-32767>
				#define Spd_Kt 			32767
				
			// <o> MaxLimit parameters (unit : mA)) <0-32767>
				#define Spd_MaxLimit 	(int16)((float) 2500/1000 * I_AMPLIFIER)
				
			// <o> MinLimit parameters (unit : mA) <0-32767>
				#define Spd_MinLimit 	(int16)((float) -0/1000 * I_AMPLIFIER)
				
			// <o> Speed Cycle parameters <0-255>
				#define Spd_Cyc 		5
		// </h>
	// </h>

	// <h> PLL
		// <h> Set PLL PI parameters
			// <o> TailWind Kp parameters <0-32767>
				#define Pll_Kp_TailWind 31000
				
			// <o> HeadWind Kp parameters <0-32767>
				#define Pll_Kp_HeadWind 0
				
			// <o> Start Kp parameters <0-32767>
				#define Pll_Kp_Start 4000
				
			// <o> Final Kp parameters <0-32767>
				#define Pll_Kp_Final 31000
				
			// <o> Ki parameters <0-32767>
				#define Pll_Ki 10
				
			// <o> Kt parameters <0-32767>
				#define Pll_Kt 32767
				
			// <o> MaxLimit parameters <0-32767>
				#define Pll_MaxLimit 32767
				
			// <o> MinLimit parameters <0-32767>
				#define Pll_MinLimit -32767
		// </h>
	// </h>
// </h>

// <h> Set motor control program
	// <o> Set CW/CCW steering <0=> CW <1=>CCW
		#define CW_CCW 0
		
	// <o> Set Stop_Fun stop speed (unit : rpm)
	// <i> Motor stop output pwm speed
		#define STOP_SPEED_VALUE (short)((float) 60 * 32767 / (BASE_RPM))
	
	// <h> Peripheral Control MODE
		// <h> 	Remote control commands Enable/Disable
			// <h> Power Control
				// <o> 	Set IRF Button Power1 (unit : mW)<0-6000>
					#define IRF_Power_1				200
					
				// <o> Set RF Button Power2 (unit : mW)<0-6000>
					#define IRF_Power_2				400
					
				// <o> Set RF Button Power3 (unit : mW)<0-6000>
					#define IRF_Power_3				600

				// <o> Set RF Button Power4 (unit : mW)<0-6000>
					#define IRF_Power_4				800
						
				// <o> Set RF Button Power5 (unit : mW)<0-6000>
					#define IRF_Power_5				1000
			// </h>
			
			// <h> Speed Control
				// <o> 	Set IRF Button SPEED1 speed
					#define IRF_SPEED_1					(short)((float) 180 * 32767 / (BASE_RPM)) //175
				
				// <o> Set IRF Button SPEED2 speed
					#define IRF_SPEED_2					(short)((float) 240 * 32767 / (BASE_RPM)) //235
				
				// <o> Set IRF Button SPEED3 speed
					#define IRF_SPEED_3					(short)((float) 285 * 32767 / (BASE_RPM)) //280

				// <o> Set IRF Button SPEED4 speed
					#define IRF_SPEED_4					(short)((float) 335 * 32767 / (BASE_RPM)) //330
				
				// <o> Set IRF Button SPEED5 speed
					#define IRF_SPEED_5					(short)((float) 375 * 32767 / (BASE_RPM)) //370

				// <o> Set IRF Button Boost speed
					#define IRF_SPEED_BOOST				(short)((float) 415 * 32767 / (BASE_RPM)) //500
			
				// <e> Observe Speed_UI parameters Enable
					#define	 Observe_Speed_UI		0
				// </e>
			
				// <o> Set IRF Button1_SPEED_UI<0-32767>
					#define IRF_SPEED_UI_1				500
				
				// <o> Set IRF Button2_SPEED_UI<0-32767>
					#define IRF_SPEED_UI_2				700
				
				// <o> Set IRF Button3_SPEED_UI<0-32767>
					#define IRF_SPEED_UI_3				1100
				
				// <o> Set IRF Button4_SPEED_UI<0-32767>
					#define IRF_SPEED_UI_4				1500
				
				// <o> Set IRF Button5_SPEED_UI<0-32767>
					#define IRF_SPEED_UI_5				1900

				// <o> Set IRF ButtonBOOST_SPEED_UI<0-32767>
					#define IRF_SPEED_UI_BOOST			2500
				
			// </h>
			
			// <h> Current Control
				// <o> 	Set RF Button Current1 (unit : mA)<300-4000>
					#define IR_Current_1_VALUE 			(int16)((float)  775/1000 * I_AMPLIFIER) // unit : A * Gain
					
				// <o> Set RF Button Current2 (unit : mA)<300-4000>
					#define IR_Current_2_VALUE 			(int16)((float)  1210/1000 * I_AMPLIFIER) // unit : A * Gain
					
				// <o> Set RF Button Current3 (unit : mA)<300-4000>
					#define IR_Current_3_VALUE 			(int16)((float) 1560/1000 * I_AMPLIFIER) // unit : A * Gain

				// <o> Set RF Button Current4 (unit : mA)<300-4000>
					#define IR_Current_4_VALUE 			(int16)((float) 1850/1000 * I_AMPLIFIER) // unit : A * Gain
						
				// <o> Set RF Button Current5 (unit : mA)<300-4500>
					#define IR_Current_5_VALUE 			(int16)((float) 2070/1000 * I_AMPLIFIER) // unit : A * Gain
				
			// </h>
				// <e> Power On Default State Enable/Disable
					#define Default_State 1
					#if (Default_State == 1)
						// <o> Fan_Gear_Set Default State <0=> 1 <1=> 2 <3=> 3 <4=> 4 <5=> 5
							#define  SpeedNum_Default 3
					#endif
				// </e>
		// </h>
	
		// <e> VSP control commands Enable/Disable
			#define VSP_TRI 0
			#if (VSP_TRI == 1)
				// <o> Set VSP_CH <0=> CH0 <1=> CH1 <2=> CH2 <3=> CH3 <4=> CH4 <5=> CH5 <6=> CH6 <7=> CH7 
					#define VSP_CH 4
				// <e> Slope control
					#define dSlope_control 0
					#if (dSlope_control == 1)
					// <o> Set VSP srart voltage (unit : 0.1v)
						#define  CMD_START		(float)5/10
						#define  CMD_START_VAL	(unsigned int)((float)(CMD_START) *1023 / 5.0)
					// <o> Set VSP max voltage (unit : 0.1v)
						#define  CMD_MAX		(float)45/10
						#define  CMD_MAX_VAL	(unsigned int)((float)(CMD_MAX) *1023 / 5.0)
					// <o> Set VSP stop voltage (unit : 0.1v)
						#define  CMD_STOP		(float)3/10
						#define  CMD_STOP_VAL	(unsigned int)((float)(CMD_STOP) *1023 / 5.0)
						
						#define SPEED_STEP		((unsigned long)1024 * ((SPEED_MAX_LIMIT_VALUE) - (SPEED_MIN_LIMIT_VALUE))) / ((CMD_MAX_VAL) - (CMD_START_VAL))
						//1023 * (2800 / (920.7 - 102.3)  
						//1023 * 2800 / 818.4
					#endif
				// </e>
			
				// <e> Stage control
					#define dStage_control 0
					#if (dStage_control == 1)
					// <o> 	Set VSP SPEED1 speed
						#define VSP_SPEED_1					(short)((float) 100 * 32768 / BASE_RPM)
						
					// <o> Set VSP SPEED2 speed
						#define VSP_SPEED_2					(short)((float) 150 * 32768 / BASE_RPM)
						
					// <o> Set VSP SPEED3 speed
						#define VSP_SPEED_3					(short)((float) 180 * 32768 / BASE_RPM)

					// <o> Set VSP SPEED4 speed
						#define VSP_SPEED_4					(short)((float) 200 * 32768 / BASE_RPM)
					
					// <o> Set VSP_SPEED_UI_1<0-32767>
						#define VSP_SPEED_UI_1				350
					
					// <o> Set VSP_SPEED_UI_2<0-32767>
						#define VSP_SPEED_UI_2				800
						
					// <o> Set VSP_SPEED_UI_3<0-32767>
						#define VSP_SPEED_UI_3				1000
						
					// <o> Set VSP_SPEED_UI_4<0-32767>
						#define VSP_SPEED_UI_4				1200
					
					#endif
				// </e>
				
				#if (dSlope_control == 1) && (dStage_control == 1)
					#error Wrong setting Slope_control and Stage_control !!!!
				#endif
			#endif
		// </e>
					
				// <h> IQ OUTMAX
					// <o> Set IQOUTMAX (Ibus_avg)
					#define IQOUTMAX  4000/100   //7400
				// </h> 
		// <e> Constant current control Enable/Disable
			#define CURRENT_CONTROL 0
		
			#if (CURRENT_CONTROL == 1)
			#endif
				// <o> Set current control range(Max) (unit : mA)
					#define IQ_MAX_LIMIT_VALUE (int16)((float) 2200/1000 * I_AMPLIFIER) // unit : A * Gain

				// <o> Set current control range(Min) (unit : mA)
					#define IQ_MIN_LIMIT_VALUE (int16)((float) 0/1000 * I_AMPLIFIER) // unit : A * Gain
					
					#define IQ_GAIN ((float) (IQ_MAX_LIMIT_VALUE) / 1023)
					
				// <o> Set CURRENT_RAMP slope (unit : ms)
					#define CURRENT_RAMP		20
					
				// <o> Set CURRENT_CMD accelerate RAMP
					#define CURRENT_ACC			5
					
				// <o> Set CURRENT_CMD decelerate RAMP
					#define CURRENT_DEC			10
		// </e>
		
		// <e> Constant speed control Enable/Disable
			#define SPEED_CONTROL 1
			#if (SPEED_CONTROL == 1)
			#endif
				// <o> Set speed control range(Max) (unit : rpm)
					#define SPEED_MAX_LIMIT_VALUE (short)((float) 420 * 32767 / (BASE_RPM))    // 2960, BASE_RPM = 4095
					
				// <o> Set speed control range(Min) (unit : rpm)
					#define SPEED_MIN_LIMIT_VALUE (short)((float) 150 * 32767 / (BASE_RPM)) 
					
				// <o> Set SPEED_RAMP slope (unit : ms)
					#define SPEED_RAMP		20
					
				// <o> Set SPEED_CMD accelerate RAMP
					#define SPEED_ACC		60
					
				// <o> Set SPEED_CMD decelerate RAMP
					#define SPEED_DEC		100
		// </e>
		#if (CURRENT_CONTROL == 1) && (SPEED_CONTROL == 1)
			#error Wrong setting CURRENT_CONTROL and SPEED_CONTROL !!!!
		#endif
		// <e> Ibus Limit Enable/Disable
			#define IBUS_LIMIT 0
			#if (IBUS_LIMIT == 1)
				// <o> Ibus limit values (unit : 0.1A) <0-100>
					#define IBUS_LIMIT_VALUE 	 (float)35/10
				// <o> Ibus OPA GAIN <0-100>
					#define IBUS_OPA_GAIN 	 	 11
				// <o> Ibus Correction factor  <0-1000>
					#define IBUS_COFF 	 		 (float)100/100
						
					#define	dIDC_GAIN_W			 ((float)((float)(0.01 * (IBUS_OPA_GAIN) * 1023) / 5) * 64 * (IBUS_COFF))
			#endif
		// </e>
		// <e> Power Limit Enable/Disable
			#define POWER_LIMIT 1
			#if (POWER_LIMIT == 1)
				// <o> Input power limit (unit : 0.01W) <0-32767>
				// <i> Note: Fill in the input Vbus_avg, Ibus_avg multiplied, WATT value
					#define WATT_LIMIT_VALUE (uint32) 3636
					
				// <o> Input power limit Execution cycle (unit : ms)
					#define POWER_LIMIT_TIMER 30
					
				// <o> Set power Limit delay time (unit : ms)
					#define POWER_LIMIT_DELAY_DURATION 10
					
				// <o> Set I_BUS_CH <0=> CH0 <1=> CH1 <2=> CH2 <3=> CH3 <4=> CH4 <5=> CH5 <6=> CH6 <7=> CH7 
					#define I_BUS_CH 2
					
				// <o> Set Power rate parameter (unit : 0.0001)
					#define dPOWER_GAIN ((float) 389/10000)
					
			#endif
		// </e>
			
		// <e> Power Control Enable/Disable
			#define POWER_CONTROL 0
			#if (POWER_CONTROL == 1)
				// <o> Input Watt parameters (unit : 0.01W) <0-32767>
				// <i> Note : Fill in the input Vbus_avg and Ibus_avg after multiplying, the WATT value (Q15 format)
					#define WATT_LIMIT_VALUE (uint32) 3550
				
				// <o> Input power Control execution cycle (unit : ms)
					#define POWER_CONTROL_DELAY_DURATION 2
					
				// <o> Set I_BUS_CH <0=> CH0 <1=> CH1 <2=> CH2 <3=> CH3 <4=> CH4 <5=> CH5 <6=> CH6 <7=> CH7 
					#define I_BUS_CH 2
					
				// <o> Set First Power_PI_Out To IQ_Cmd current (unit : mA) 
					#define POWER_PI_OUT_VALUE (int16)((float) 2000/1000 * I_AMPLIFIER) // unit : A * Gain
					
				// <o> Set power magnification parameters (unit : 0.00001)
				// <i> Power_PI_Out To IQ_Cmd
					#define dPOWER_GAIN ((float)430/10000) 
					
				// <o> Power control tuning process <1=> LEVEL 1 <2=> LEVEL 2
				// <i> LEVEL_1 : Not executed PowerControl_Fun
				// <i> LEVEL_2 : execute program
					#define POWER_CONTROL_SOP 2
			#endif
		// </e>

		#if (POWER_LIMIT == 1) && (POWER_CONTROL == 1)
			#error Wrong setting POWER_LIMIT and POWER_CONTROL !!!!
		#endif

		// <e> CW/CCW test Enable/Disable
			#define CW_CCW_TEST	0
			#if (CW_CCW_TEST == 1)
				// <o> Running time after changing direction (unit : ms) 
					#define		Reversing_TIMES		8000
					
				// <o> Stop this time before catching the wind (unit : ms)
					#define		STOP_TIMES_TO_CW	2000
					
				// <o> Operating time after connecting downwind (unit : ms)
					#define		CW_RUN_TIME			50000
			#endif
		// </e>

		// <e> IPD Enable/Disable
			#define IPD_FUNCTION 1
			// <e> Automatic cycle test of initial position Enable/Disable
				#define IPD_AUTO_TEST 0
			// </e>
		// </e>

		// <e> Brake control Enable/Disable
			#define BREAK_FUNCTION 1
			#if (BREAK_FUNCTION == 1)
				// <o> Set braking force (unit : %) <0-100>
				// <i> Description: Set the braking force in the tailwind state
					#define BREAK_DUTY (float)99/100
					#define BREAK_DUTY_VALUE (uint16)((float) (MPWMDATA_REGS) * (BREAK_DUTY))
			#else
				#define BREAK_DUTY (float)99/100
				#define BREAK_DUTY_VALUE (uint16)((float) (MPWMDATA_REGS) * (BREAK_DUTY))
			#endif
		// </e>

		// <e> Bumpless control Enable/Disable
			#define BUMPLESS_FUNCTION 1
			#if (BUMPLESS_FUNCTION == 1)
				// <o> BUMPLESS_DURATION parameter <0-65535>
					#define BUMPLESS_DURATION 100
			#endif
		// </e>
	// </h>
// </h>
				
// <h> Set Fairwind and Headwind judgment function
	// <e> BEMF Fairwind/Headwind judgment (resistance) Enable/Disable
		#define BEMF_TAILWIND_RES_FUNCTION 1
		#if (BEMF_TAILWIND_RES_FUNCTION == 1)
			// <o> BEMF Fairwind/Headwind adjustment process <1=> LEVEL 1 <2=> LEVEL 2
			// <i> LEVEL_1 : Not executed IPDStart_Fun, Stop the program to TailWindDetect_Fun, TailWindDetect_Bemf_Fun, TailWindDetect_OneBemf_Fun, TailWindDetect_TwoBemf_Fun 
			// <i> LEVEL_2 : execute program
				#define BEMF_TAILWIND_SOP 2
			
			// <h> Set BEMF A/D Channel
			
				// <o> Set BEMF_V_CH <0=> CH0 <1=> CH1 <2=> CH2 <3=> CH3 <4=> CH4 <5=> CH5 <6=> CH6 <7=> CH7 
					#define BEMF_V_CH 3
					
				// <o> Set BEMF_W_CH <0=> CH0 <1=> CH1 <2=> CH2 <3=> CH3 <4=> CH4 <5=> CH5 <6=> CH6 <7=> CH7 
					#define BEMF_W_CH 5
			// </h>
			
			// <h> Set Speed for judging Fairwind/Headwind
			
				// <o1> Set the minimum connection speed for judging Fairwind 	(unit : rpm)
					#define	DET_Tailspeed_min_speed		(short)((long)93750/((10*  (POLE_PAIRS))/60))
			
				// <o1> Set the maximum connection speed for judging Fairwind 	(unit : rpm)
					#define	DET_Tailspeed_max_speed		(short)((long)93750/((180* (POLE_PAIRS))/60))	
			
				// <o1> Set the maximum connection speed to judge the headwind 	(unit : rpm)	
					#define	DET_Headspeed_max_speed		(short)((long)93750/((100*  (POLE_PAIRS))/60))	//87
				
				// <o> Set the duration of Fairwind start judgment (unit : ms)
					#define TAILWIND_DURATION 50 
				
				// #define SPEED_FINAL_START_VALUE  (short)(((long) 10 * POLE_PAIRS) >> AngleBaseShift)
				// #define SPEED_FINAL_START_VALUE1 (short)(((long) 150 * POLE_PAIRS) >> AngleBaseShift)
				// #define SPEED_FINAL_START_VALUE2 (short)(((long) 250 * POLE_PAIRS) >> AngleBaseShift)
				
			// </h>
			
			// <o> Set Headwind_high_speed_start_Strategy <1=>PLL <2=>ABS
				#define Headwind_High_Speed_Start_Strategy 2

			// <o> Set Fairwind/Headwind angle(CW_CCW=0) <0-383>
			// <i> 0~383 = 0~360 degrees
					#define CW_TAILWIND_RES_ANG 	0
					
			// <o> Set Fairwind/Headwind angle(CW_CCW=1) <0-383>	
			// <i> 0~383 = 0~360 degrees
					#define CCW_TAILWIND_RES_ANG 	64
			
			// <o> Set IQ Fairwind/Headwind starting current(IQ_CMD) (unit : mA)
				#define IQ_FINAL_TAILWIND_START (float)2000/1000 
				#define IQ_FINAL_TAILWIND_START_VALUE (int16)((float) (IQ_FINAL_TAILWIND_START) * (I_AMPLIFIER)) // unit : A * Gain
			
			// <o> Set ID Fairwind/Headwind starting current(ID_CMD) (unit : mA)
				#define ID_FINAL_TAILWIND_START (float)0/1000
				#define ID_FINAL_TAILWIND_START_VALUE (int16)((float) -(ID_FINAL_TAILWIND_START) * (I_AMPLIFIER)) // unit : A * Gain
				
			// <o> Set the initial value of Fairwind/Headwind current(IQ_UI) (unit : Val) <0-60000>
				#define BEMF_TAILWIND_IQ_OUT_VALUE (short)(8000)
			
			// <o> Set Headwind to Fairwind target speed (unit : rpm) <0-60000>
			// <i> 	Determine the target speed from headwind to Fairwind
				#define BEMF_TAILWIND_SPEED_VALUE (short)((float) 50 * 32767 / (BASE_RPM))
			
			// <e> Observe BEMF_V_W Parameters
				#define Observe_BEMF				0
			// </e>
			// <o> Set the judgment counter for the static BEMF value of the motor
				#define dMotor_standstill_judgment_cnt	300
		#endif
	// </e>

	// <e> BEMF Fairwind/Headwind judgment (Diode) Enable/Disable
	// <i> Note: Requires Enable brake control function
		#define BEMF_TAILWIND_DIODE_FUNCTION 0
		#if (BEMF_TAILWIND_DIODE_FUNCTION == 1)	
			// <o> BEMF Fairwind/Headwind adjustment process <1=> LEVEL 1 <2=> LEVEL 2
			// <i> LEVEL_1 : Not executed IPDStart_Fun, Stop the program to TailWindDetect_Fun, TailWindDetect_Bemf_Fun, TailWindDetect_OneBemf_Fun, TailWindDetect_TwoBemf_Fun 
			// <i> LEVEL_2 : execute program
				#define BEMF_TAILWIND_SOP 2
			
			// <h> Set BEMF A/D Channel
			
				// <o> Set BEMF_V_CH <0=> CH0 <1=> CH1 <2=> CH2 <3=> CH3 <4=> CH4 <5=> CH5 <6=> CH6 <7=> CH7 
					#define BEMF_V_CH 4
					
				// <o> Set BEMF_W_CH <0=> CH0 <1=> CH1 <2=> CH2 <3=> CH3 <4=> CH4 <5=> CH5 <6=> CH6 <7=> CH7 
					#define BEMF_W_CH 5
			
			// </h>
			
			// <h> Set Speed for judging Fairwind/Headwind
			
				// <o1> Set the minimum connection speed for judging Fairwind 	(unit : rpm)
					#define	DET_Tailspeed_min_speed		(short)((long)93750/((23*  (POLE_PAIRS))/60))
			
				// <o1> Set the maximum connection speed for judging Fairwind 	(unit : rpm)
					#define	DET_Tailspeed_max_speed		(short)((long)93750/((140* (POLE_PAIRS))/60))	
			
				// <o1> Set the maximum connection speed to judge the headwind 	(unit : rpm)	
					#define	DET_Headspeed_max_speed		(short)((long)93750/((100*  (POLE_PAIRS))/60))	//87
				
				// <o> Set the duration of Fairwind start judgment (unit : ms)
					#define TAILWIND_DURATION 100 
				
				// #define SPEED_FINAL_START_VALUE  (short)(((long) 10 * POLE_PAIRS) >> AngleBaseShift)
				// #define SPEED_FINAL_START_VALUE1 (short)(((long) 150 * POLE_PAIRS) >> AngleBaseShift)
				// #define SPEED_FINAL_START_VALUE2 (short)(((long) 250 * POLE_PAIRS) >> AngleBaseShift)
				
			// </h>
			
			// <o> Set Fairwind/Headwind angle(CW_CCW=0) <0-383>
			// <i> 0~383 = 0~360 degrees
					#define CW_TAILWIND_DIODE_ANG 	96
					
			// <o> Set Fairwind/Headwind angle(CW_CCW=1) <0-383>
			// <i> 0~383 = 0~360 degrees
					#define CCW_TAILWIND_DIODE_ANG 	128
					
			// <o> Set IQ Fairwind/Headwind starting current(IQ_CMD) (unit : mA)
				#define IQ_FINAL_TAILWIND_START (float)2000/1000 
				#define IQ_FINAL_TAILWIND_START_VALUE (int16)((float) (IQ_FINAL_TAILWIND_START) * (I_AMPLIFIER)) // unit : A * Gain
				
			// <o> Set the initial value of Fairwind/Headwind current(IQ_UI) (unit : Val) <0-60000>
				#define BEMF_TAILWIND_IQ_OUT_VALUE (short)(4000)
				
			// <o> Set ID Fairwind/Headwind starting current(ID_CMD) (unit : mA)
				#define ID_FINAL_TAILWIND_START (float)0/1000
				#define ID_FINAL_TAILWIND_START_VALUE (int16)((float) -(ID_FINAL_TAILWIND_START) * (I_AMPLIFIER)) // unit : A * Gain
				
			// <o> Set Headwind to Fairwind target speed (unit : rpm) <0-60000>
				#define BEMF_TAILWIND_SPEED_VALUE (short)((float) 200 * 32767 / (BASE_RPM))
		#endif
	// </e>
	// <e> Fairwind/Headwind judgment (Estimator) Enable/Disable
		#define TAILWIND_ESTIMATED_FUNCTION 0
		#if (TAILWIND_ESTIMATED_FUNCTION == 1)
		// <o> Set Fairwind speed (unit : rpm) <0-1000>
			#define SPEED_TAILWIND_VALUE (short)((float) 20 * 32767 / (BASE_RPM))
			
		// <o> Set Fairwind/Headwind speed (unit : rpm) <0-60000>
		// <i> Description: Set the speed when connected in a Fairwind state
			#define BEMF_TAILWIND_SPEED_VALUE (short)((float) 200 * 32767 / (BASE_RPM))
			
		// <o> BEMF Fairwind/Headwind adjustment process <1=> LEVEL 1 <2=> LEVEL 2
		// <i> LEVEL_1 : Not executed IPDStart_Fun, Stop the program to TailWindDetect_Fun, TailWindDetect_Bemf_Fun, TailWindDetect_OneBemf_Fun, TailWindDetect_TwoBemf_Fun 
		// <i> LEVEL_2 : execute program	
			#define BEMF_TAILWIND_SOP 2
		#endif
	// </e>

	#if ((TAILWIND_ESTIMATED_FUNCTION == 0) && (BEMF_TAILWIND_RES_FUNCTION == 0) && (BEMF_TAILWIND_DIODE_FUNCTION == 0))
		#define BEMF_TAILWIND_SPEED_VALUE 0
	#endif
// </h>

// <h> Set motor protection function
	// <e> Overvoltage/Undervoltage protection (OVP/UVP)
	// <i> Note: Need to use Vbus voltage divider resistors
	// <i> (OVER) (CLEAR_OVER)                         (CLEAR_UNDER) (UNDER)
	// <i> ----|-----------|------------------------------|-------------|---->
		#define Vbus_Protect 1
		#if (Vbus_Protect == 1)
			// <o> Set V_BUS_CH <0=> CH0 <1=> CH1 <2=> CH2 <3=> CH3 <4=> CH4 <5=> CH5 <6=> CH6 <7=> CH7 
				#define V_BUS_CH 6
				
			// <o> Set Vbus rate parameter (unit : 0.001)
			// <i> VBUS_GAIN = (Vbus_avg / Input voltage)*1000
				#define VBUS_GAIN ((float) 27739 / 1000)    // (R4=30K, R8=4.7K), 30V*(R8/(R4+R8))*(1024/5) = 832.18, 832.18/30 = 27.739
				
			// <o> OVER_BUS_VOLT_VALUE parameter (unit : 0.1V)
				#define OVER_BUS_VOLT_VALUE (uint16)((float) 350 * 0.1 * (VBUS_GAIN)) // unit : V * Gain
				
			// <o> CLEAR_OVER_BUS_VOLT_VALUE parameter (unit : 0.1V)
				#define CLEAR_OVER_BUS_VOLT_VALUE (uint16)((float) 295 * 0.1 * VBUS_GAIN) // unit : V * Gain
				
			// <o> CLEAR_UNDER_BUS_VOLT_VALUE parameter (unit : 0.1V)
				#define CLEAR_UNDER_BUS_VOLT_VALUE (uint16)((float) 280 * 0.1 * VBUS_GAIN) // unit : V * Gain
				
			// <o> UNDER_BUS_VOLT_VALUE parameter (unit : 0.1V)
				#define UNDER_BUS_VOLT_VALUE (uint16)((float) 270 * 0.1 * VBUS_GAIN) // unit : V * Gain
				
			// <o> BUS_VOLT_DURATION parameter (unit : ms)
				#define BUS_VOLT_DURATION 500 // unit : mSec
		#endif
	// </e>

	// <e> Locked-rotor protection
		// <i> Description: Speed Iq Id feedback abnormal related
		#define FaultLock_Protect 1
		#if (FaultLock_Protect == 1)
			// <o> The speed is too high abnormal (unit : rpm)
				#define OverSpeed (short)((float) 430 * 32767 / (BASE_RPM))
				
			// <o> The speed is too low abnormal (unit : rpm)
				#define UnderSpeed (short)((float) 5 * 32767 / (BASE_RPM))
				
			// <o> Locked-rotor protection judgment delay time (unit : 1ms)
				#define FAULTLOCK_DELAY_DURATION 200
		#endif
	// </e>
	
	// <e> Over temperature protection(OTP)
	// <i> Note: Need to use temperature sensor
	// <i> (OVER) (CLEAR_OVER)                         (CLEAR_UNDER) (UNDER)
	// <i> ----|-----------|------------------------------|-------------|---->
		#define Temperature_Protect 0
		#if (Temperature_Protect == 1)
		// <o> Set OTP_CH <0=> CH0 <1=> CH1 <2=> CH2 <3=> CH3 <4=> CH4 <5=> CH5 <6=> CH6 <7=> CH7 
			#define TEMPERATURE_CH 3
			
		// <o> OVER_TEMPERATURE parameter (unit : Val)
			#define OVER_TEMPERATURE_VALUE 900 // unit : DegC * Gain
			
		// <o> CLEAR_OVER_TEMPERATURE parameter (unit : Val)
			#define CLEAR_OVER_TEMPERATURE_VALUE 800 // unit : DegC * Gain
			
		// <o> TEMPERATURE_DURATION parameter (unit : ms)
			#define TEMPERATURE_DURATION 50 // unit : mSec
		#endif
	// </e>
	
	// <e> Phase current protection
	// <i> Description : Ia,Ib,Ic
		#define POCP_Protect 0
		#if (POCP_Protect == 1)
			// <o> Phase Ia,Ib,Ic parameter (unit : mA)
				#define PHASE_OCP (float)700/1000 
				#define PHASE_OCP_VALUE (int16)((float) PHASE_OCP * I_AMPLIFIER) // unit : A * Gain
				
			// <o> PHASE_OCP_DURATION parameter (unit : ms)
				#define PHASE_OCP_DURATION 50
		#endif
	// </e>
	
	
							
	// <h> Set Protection to retry
		// <e> OVP_Retry_Enable
			#define OVP_Retry_ENABLE			1
		// </e>
		
		// <e> LVP_Retry_Enable
			#define LVP_Retry_ENABLE			0
		// </e>
		
		// <e> AOCP_Retry_Enable
			#define AOCP_Retry_ENABLE			1
		// </e>
		
		// <e> POCP_Retry_Enable
			#define POCP_Retry_ENABLE			0
		// </e>
		
		// <e> FaultLock_Retry_Enable
			#define FaultLock_Retry_ENABLE		1
		// </e>
		
		// <e> OTP_Retry_Enable
			#define OTP_Retry_ENABLE			0
		// </e>
		
		// <o> Set Retry times <0-255>
			#define RETRY_COUNT 5
		// <o> Set Restart delay time(unit : ms) <0-65535>
			#define RESTART_DURATIONT 1000
			
	// </h>
// </h>

// <h> Error code(MotorErrorState)
	// <i> 	Clear = 			0(Normal)
	// <i>	OverVbus = 			1(Over voltage)
	// <i>	UnderVbus = 		2(Under voltage)
	// <i>	OverTemperature = 	4(Over temperature)
	// <i>	FaultLock = 		8(Stall)
	// <i>	AOCP = 				16(Over current)
	// <i>	POCP = 				32(phase current fault)
	
// </h>

// <h> Motor state(MotorState)
	// <i>	M_OFF = 			0(Error restart status)
	// <i>	M_INIT = 			1(Initial value setting)
	// <i>	M_TAILWIND = 		2(Forward and backward wind detection)
	// <i>	M_IPD = 			3(Initial angle detection)
	// <i>  M_Parking =			4(Positioning Strategy)
	// <i>	M_START = 			5(Motor start process)
	// <i>	M_RUN = 			6(Entering the open-loop control)
	// <i>	M_STOP = 			7(Motor stop)
	// <i>	M_ERROR = 			8(No longer restarting after an error)
	// <i>	M_BMF_BREAK = 		9(Brake detection BEMF status)
// </h>
// <<< end of configuration section >>>
//*******************************************************************************************************************************************
// MOTOR_CONT1
#define MPWMSEL 0x40
#define MPWMEN 0x20
#define IQINSEL 0x10
#define FOCANGSEL 0x08

// MOTOR_CONT2
#define MPWMCPSEN 0x80
#define MPWMCPSTP 0x00
//#define MPWMCPSTP 0x20
//#define MPWMCPSTP 0x60
#define SVPWMPS 0x08 // CW or CCW
#define CPS_LIMIT_EN 0x00
//#define SVPWMMODE 0x00 // SVPwm
#define SVPWMMODE 0x10 // DPwm

// MOTOR_CONT3
#define SMO_CMP_VALUE 0x40

// PI_GAIN
#define PLLKIGEN 0x01
#define PLLKPGEN 0x02
#define SPEEDKIGEN 0x04
#define SPEEDKPGEN 0x08
#define IDKIGEN 0x10
#define IDKPGEN 0x20
#define IQKIGEN 0x40
#define IQKPGEN 0x80

// PI_TMSR
#define PLLTMEN 0x01
#define PLLTMSEL 0x02
#define SPEEDTMEN 0x04
#define SPEEDTMSEL 0x08
#define IDTMEN 0x10
#define IDTMSEL 0x20
#define IQTMEN 0x40
#define IQTMSEL 0x80

//FOCCONT
#define SPEEDEN 0x01
#define PLLEN 0x02
#define ADCTRIG 0x04
#define INVADCD 0x08
#define ESTCR 0x00
#define PICLEAR 0x80

#define USER_FB_MACRO(v) SFR_PAGE = 4; PI_FB = v
#define GET_USER_OUT_MACRO(v) SFR_PAGE = 4; v = PI_OUT

#define IQ_CMD_MACRO(v) SFR_PAGE = 0; PI_CMD = v
#define GET_IQ_OUT_MACRO(v) SFR_PAGE = 0; v = PI_OUT
#define ID_CMD_MACRO(v) SFR_PAGE = 1; PI_CMD = v

#define SPEED_CMD_MACRO(v) SFR_PAGE = 2; PI_CMD = v
#define PLL_CMD_MACRO(v) SFR_PAGE = 3; PI_CMD = v
#define USER_CMD_MACRO(v) SFR_PAGE = 4; PI_CMD = v

#define IQ_KP_MACRO(v) SFR_PAGE = 0; PI_KP = v
#define ID_KP_MACRO(v) SFR_PAGE = 1; PI_KP = v
#define SPEED_KP_MACRO(v) SFR_PAGE = 2; PI_KP = v
#define PLL_KP_MACRO(v) SFR_PAGE = 3; PI_KP = v
#define USER_KP_MACRO(v) SFR_PAGE = 4; PI_KP = v

#define IQ_KI_MACRO(v) SFR_PAGE = 0; PI_KI = v
#define ID_KI_MACRO(v) SFR_PAGE = 1; PI_KI = v
#define SPEED_KI_MACRO(v) SFR_PAGE = 2; PI_KI = v
#define PLL_KI_MACRO(v) SFR_PAGE = 3; PI_KI = v
#define USER_KI_MACRO(v) SFR_PAGE = 4; PI_KI = v

#define IQ_KT_MACRO(v) SFR_PAGE = 0; PI_KT = v
#define ID_KT_MACRO(v) SFR_PAGE = 1; PI_KT = v
#define SPEED_KT_MACRO(v) SFR_PAGE = 2; PI_KT = v
#define PLL_KT_MACRO(v) SFR_PAGE = 3; PI_KT = v
#define USER_KT_MACRO(v) SFR_PAGE = 4; PI_KT = v

#define IQ_TR_MACRO(v) SFR_PAGE = 0; PI_KT = v
#define ID_TR_MACRO(v) SFR_PAGE = 1; PI_KT = v
#define SPEED_TR_MACRO(v) SFR_PAGE = 2; PI_KT = v
#define PLL_TR_MACRO(v) SFR_PAGE = 3; PI_KT = v
#define USER_TR_MACRO(v) SFR_PAGE = 4; PI_KT = v

#define IQ_UI_MACRO(v) SFR_PAGE = 0; PI_UI = v
#define ID_UI_MACRO(v) SFR_PAGE = 1; PI_UI = v
#define SPEED_UI_MACRO(v) SFR_PAGE = 2; PI_UI = v
#define PLL_UI_MACRO(v) SFR_PAGE = 3; PI_UI = v
#define USER_UI_MACRO(v) SFR_PAGE = 4; PI_UI = v

#define IQ_OUT_MACRO(v) SFR_PAGE = 0; PI_OUT = v
#define ID_OUT_MACRO(v) SFR_PAGE = 1; PI_OUT = v
#define SPEED_OUT_MACRO(v) SFR_PAGE = 2; PI_OUT = v
#define PLL_OUT_MACRO(v) SFR_PAGE = 3; PI_OUT = v
#define USER_OUT_MACRO(v) SFR_PAGE = 4; PI_OUT = v

#define IQ_MAX_MACRO(v) SFR_PAGE = 0; PI_MAX = v
#define ID_MAX_MACRO(v) SFR_PAGE = 1; PI_MAX = v
#define SPEED_MAX_MACRO(v) SFR_PAGE = 2; PI_MAX = v
#define PLL_MAX_MACRO(v) SFR_PAGE = 3; PI_MAX = v
#define USER_MAX_MACRO(v) SFR_PAGE = 4; PI_MAX = v

#define IQ_MIN_MACRO(v) SFR_PAGE = 0; PI_MIN = v
#define ID_MIN_MACRO(v) SFR_PAGE = 1; PI_MIN = v
#define SPEED_MIN_MACRO(v) SFR_PAGE = 2; PI_MIN = v
#define PLL_MIN_MACRO(v) SFR_PAGE = 3; PI_MIN = v
#define USER_MIN_MACRO(v) SFR_PAGE = 4; PI_MIN = v

#define GS_MACRO(v) SFR_PAGE = 0; SMO_D1 = v
#define SMO_KSLIDE_MACRO(v) SFR_PAGE = 1; SMO_D1 = v
#define ANG_BASE_MACRO(v) SFR_PAGE = 2; SMO_D1 = v
#define Z_CORR_MACRO(v) SFR_PAGE = 3; SMO_D1 = v
#define SMO_ANG_MACRO(v) SFR_PAGE = 4; SMO_D1 = v
#define BAN_GAIN_MACRO(v) SFR_PAGE = 5; SMO_D1 = v

#define FS_MACRO(v) SFR_PAGE = 0; SMO_D2 = v
#define SMO_FILTER_MACRO(v) SFR_PAGE = 1; SMO_D2 = v
#define ANG_SHIFT_MACRO(v) SFR_PAGE = 2; SMO_D2 = v
#define MAXSMC_ERR_MACRO(v) SFR_PAGE = 3; SMO_D2 = v
#define SPEED_MACRO(v) SFR_PAGE = 4; SMO_D2 = v
#define VQ_OFFSET_MACRO(v) SFR_PAGE = 0; FOC_D2 = v; SYNC = 0x55
#define VD_OFFSET_MACRO(v) SFR_PAGE = 1; FOC_D2 = v; SYNC = 0x55
#define AS_MACRO(v) SFR_PAGE = 3; FOC_D2 = v
#define CPU_ANG_MACRO(v) SFR_PAGE = 4; FOC_D2 = v; SYNC = 0x55
#define FOC_ANG_MACRO(v) SFR_PAGE = 5; FOC_D2 = v
#define SVPWM_AMP_MACRO(v) SFR_PAGE = 6; FOC_D2 = v

#define GET_IQ_UI_MACRO(v) SFR_PAGE = 0; v = PI_UI
#define GET_SPEED_UI_MACRO(v) SFR_PAGE = 2; v = PI_UI
#define GET_SPEED_MACRO(v) SFR_PAGE = 4; v = SMO_D2
#define GET_VQ_MACRO(v) SFR_PAGE = 0; v = PI_OUT;
#define GET_IQ_MACRO(v) SFR_PAGE = 0; v = PI_FB;

#define IA_AD_OFFSET(v) SFR_PAGE = 0; ADCOFST = v
#define IB_AD_OFFSET(v) SFR_PAGE = 1; ADCOFST = v

#define FG_ENABLE 		FG_CTRL |= (0x80)
#define FG_DISABLE 		FG_CTRL &= ~(0x80)
#define USER_PI_ACTIVE MOTOR_CONT1 |= 0x04

#define Uint32_Div_Uint16_MACRO(Dividend,Divisor,Answer)\
MD_MODE |= 0x02;\
MD0 = Dividend & 0xFF;\
MD1 = (Dividend & 0xFF00) >> 8;\
MD2 = (Dividend & 0xFF0000) >> 16;\
MD3 = (Dividend & 0xFF000000) >> 24;\
MD4 = Divisor & 0xFF;\
MD5 = (Divisor & 0xFF00) >> 8);\
while((MD_MODE & 0x10) == 0x10);\
Answer = MD0;\
Answer |= MD1 << 8;\
Answer |= MD2 << 16;\
Answer |= MD3 << 24

#define Uint16_Div_Uint16_MACRO(Dividend,Divisor,Answer)\
MD_MODE |= 0x02;\
MD0 = Dividend & 0xFF;\
MD1 = (Dividend & 0xFF00) >> 8;\
MD4 = Divisor & 0xFF;\
MD5 = (Divisor & 0xFF00) >> 8;\
while((MD_MODE & 0x10) == 0x10);\
Answer = MD0;\
Answer |= MD1 << 8

#define Uint16_Mult_Uint16_MACRO(MultCand,MultCator,Answer)\
MD_MODE |= 0x02;\
MD0 = MultCand & 0xFF;\
MD4 = MultCator & 0xFF;\
MD1 = (MultCand & 0xFF00) >> 8;\
MD5 = (MultCator & 0xFF00) >> 8;\
while((MD_MODE & 0x10) == 0x10);\
Answer = MD0;\
Answer |= MD1 << 8;\
Answer |= MD2 << 16;\
Answer |= MD3 << 24

extern bit 	MOTOR_PWM_ENABLE;
extern bit MOTOR_OFF_FLAG;

extern idata uint8 RF_Channel_temp;
extern idata uint8 RF_Channel_old;

extern idata uint8 SpeedNum_temp;
extern uint8 SpeedNum;
extern uint8 SpeedNum_pre;
extern int16 SpeedCmd;
extern int16 SpeedCmdTemp;
extern int16 CurrentCmd;
extern uint8 MotorDir;

extern uint8 MotorState;
extern uint8 MotorErrorState;
extern uint8 MotorFaultState;

extern uint8 Timer1Count_1s;
extern uint8 Timer1Smart_1s;
extern uint16 Timer1Count_1h;
extern uint16 Timer1Smart_2h;
extern uint8 Timer1Count_user_timer;
extern uint8 Counter_timer;

extern int16 CurrentCmdLimit;
//extern uint16 IbusTemp;
extern uint16 Vbus_avg;
extern uint16 Vbus_avg_old;

extern int16 Ibus_tmp;
extern int16 Ibus_avg;
extern	idata	uint16	Ibus_avg_FILTED;
extern 	idata	float	Ibus_In_AMPS;
//extern uint16 VbusOffset;
extern uint16 IbusOffset;
extern uint16 VspTemp;
extern uint16 Vsp_avg;

extern uint16 TemperatureTemp;
extern uint16 Temperature_avg;

extern xdata int16 PllOutTemp;
extern xdata int16 IqOutTemp;
extern xdata int16 IdOutTemp;

extern xdata int16 IqPiOut;
extern xdata int16 IqFb;

extern xdata int16 LatestTheta;
extern xdata uint8 CCWFlag;
extern idata int16 EstimatedSpeed;
extern xdata int16 EstimatedSpeedOld;
extern xdata uint16 RecHallUEdgeCnt;
extern xdata uint16 HallUEdgeCnt;

extern xdata uint16 Watt;
extern xdata uint16 Watt_InRef;
extern xdata signed int	POWER_TARGET_VALUES;
extern xdata unsigned char	vbus_high_sw;

extern xdata int16 BmfVW_Sub;
extern xdata uint16 abs_BmfVW_Sub;
extern xdata int16 BmfVW_Sum_Pos_ZeroPoint;
extern xdata int16 BmfVW_Sum_Neg_ZeroPoint;
extern xdata int16 BmfU;
extern xdata int16 BmfV;
extern xdata int16 BmfW;
extern xdata int8 ZeroPointCheck;
extern xdata int8 ZeroPointDebounceCnt;
extern xdata int8 ZeroPointCnt;
extern bit ZeroPointFlag;
extern bit ZeroPointFlagOld;

extern void Motor_Control (void);
extern void ResetMOC (void);
extern void MOC_Init (void);

extern struct{
	unsigned char IR_Data_Decode_Finish:1;
	unsigned char IR_Decode_Header:1;
	unsigned char External1_TRIG:1;
	unsigned char Current_offset:1;			//1 = Current correction completed
	unsigned char CloseLoopFlag:1;
	unsigned char ZeroPointFlag:1;
	unsigned char ZeroPointFlagOld:1;
	unsigned char MOTOR_PWM_ENABLE:1;		//1 = PWM output
	unsigned char MOTOR_OFF_FLAG:1;			//1 = Order the motor to stop
	unsigned char Break_flag:1;
	unsigned char Counter_Function_sw:1;	//1 = Enable the timing function
	unsigned char Smart_Function_sw:1;		//1 = Enable the SMART function
	unsigned char RF_Learning_sw:1;			//1 = RF study done
	unsigned char Buzzer_SW:1;
	unsigned char Smart_Function_eeprom:1;
	
}FLAG;

extern	xdata	unsigned int PowerCmd;
	
// Function
#if (POWER_LIMIT == 1)
extern int Temp_I;
extern xdata signed long Watt_C;
extern void PowerLimit_Fun (void);
#endif
#if (POWER_CONTROL == 1)
extern xdata signed long Watt_C;
extern void PowerControl_Fun (void);
#endif
#if (VSP_TRI == 1)
extern void Vsp_Fun (void);
#endif
#if (BEMF_TAILWIND_RES_FUNCTION == 1) || (BEMF_TAILWIND_DIODE_FUNCTION == 1)
extern void TailWindDetect_TwoBEMF_Fun (void);
extern void MotorAngle_TwoBEMF_Fun(void);
#endif

extern void MotorInit_Fun (void);
extern void MotorStart_Flow (void);
extern void MotorStartRetry_Flow (void);
extern void MotorCloseLoop_Fun (void);

extern void  RF_IR_Fun (void);
extern short Ramp_Fun (short TargetValue,SetpointValue,HighLimit,LowLimit,ADD_SetpValue,SUB_SetpValue);
extern unsigned int LERP(unsigned int g,unsigned int a,unsigned int fa, unsigned int b,unsigned int fb);
	
extern void TailWindDetect_BEMF_Fun (void);
extern void TailWindDetect_Fun (void);
extern void TailWindStart_Fun (void);

extern void IPDDetect_Fun (void);
extern void IPDStart_Fun (void);
extern void Break_Fun (void);

#endif
